<?php

namespace App\Console\Commands;

use App\Integrations\Notification\NotificationGateway;
use App\Models\AppSetting;
use App\Models\Family;
use App\Models\FamilyEvent;
use App\Models\Person;
use App\Notifications\FamilyEventReminderNotification;
use Carbon\Carbon;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class SendFamilyEventReminders extends Command
{
    protected $signature = 'events:send-reminders';

    protected $description = 'Send scheduled reminders for upcoming family events based on per-family settings';

    public function handle(NotificationGateway $gateway): int
    {
        $now = Carbon::now();
        $this->info('Running event reminders at '.$now->toDateTimeString());

        $defaultDays = (array) (AppSetting::get('event_reminder_days_default', [3]) ?? [3]);
        $defaultTime = (string) (AppSetting::get('event_reminder_time_default', '07:00') ?? '07:00');
        $currentTime = $now->format('H:i');

        Family::query()->orderBy('id')->chunk(100, function ($families) use ($defaultDays, $defaultTime, $currentTime, $now, $gateway) {
            foreach ($families as $family) {
                $settings = $family->settings ?? [];
                $days = (array) ($settings['event_reminder_days'] ?? $defaultDays);
                $time = (string) ($settings['event_reminder_time'] ?? $defaultTime);

                if ($time !== $currentTime) {
                    continue; // only send at the configured minute
                }

                $days = collect($days)->map(fn($d) => (int) $d)->filter(fn($d) => $d >= 0 && $d <= 60)->unique()->values();
                if ($days->isEmpty()) {
                    continue;
                }

                foreach ($days as $d) {
                    $from = (clone $now)->addDays($d)->startOfDay();
                    $to = (clone $now)->addDays($d)->endOfDay();

                    $events = FamilyEvent::query()
                        ->where('family_id', $family->id)
                        ->whereBetween('starts_at', [$from, $to])
                        ->get();

                    foreach ($events as $event) {
                        $this->sendRemindersForEvent($family, $event, $gateway);
                    }
                }
            }
        });

        return Command::SUCCESS;
    }

    private function sendRemindersForEvent(Family $family, FamilyEvent $event, NotificationGateway $gateway): void
    {
        $people = $family->members()->with('user')->get();
        foreach ($people as $person) {
            if ($person->user) {
                try {
                    $person->user->notify(new FamilyEventReminderNotification($family, $event));
                } catch (\Throwable $e) {
                    Log::warning('Failed to send in-app/email reminder', [
                        'user_id' => $person->user->id,
                        'event_id' => $event->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }

            $phone = is_array($person->meta ?? null) ? ($person->meta['phone'] ?? null) : null;
            if ($phone) {
                $date = optional($event->starts_at)->format('M d, Y H:i');
                $text = 'Reminder: '.$event->title.' on '.$date.' · '.$family->name;
                try {
                    $gateway->sendSms($phone, $text);
                } catch (\Throwable $e) {
                    Log::warning('Failed to send SMS reminder', [
                        'phone' => $phone,
                        'event_id' => $event->id,
                        'error' => $e->getMessage(),
                    ]);
                }
            }
        }
    }
}
