<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Integrations\Notification\NotificationGateway;
use App\Models\AppSetting;
use App\Models\Family;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class NotificationsController extends Controller
{
    public function index(Request $request)
    {
        $defaults = [
            'days' => AppSetting::get('event_reminder_days_default', [3]),
            'time' => AppSetting::get('event_reminder_time_default', '07:00'),
        ];

        $families = Family::query()->select('id', 'name')->orderBy('name')->get();

        return view('admin.notifications.index', [
            'defaults' => $defaults,
            'families' => $families,
        ]);
    }

    public function updateSettings(Request $request)
    {
        $data = $request->validate([
            'reminder_days' => ['nullable'],
            'reminder_time' => ['required', 'date_format:H:i'],
        ]);

        $daysInput = $request->input('reminder_days');
        if (is_string($daysInput)) {
            $days = array_filter(array_map('trim', explode(',', $daysInput)), fn($v) => $v !== '');
        } elseif (is_array($daysInput)) {
            $days = $daysInput;
        } else {
            $days = [];
        }

        $days = collect($days)
            ->map(fn($d) => (int) $d)
            ->filter(fn($d) => $d >= 0 && $d <= 60)
            ->unique()->values()->all();

        AppSetting::set('event_reminder_days_default', $days);
        AppSetting::set('event_reminder_time_default', $data['reminder_time']);

        return back()->with('status', 'Reminder defaults updated.');
    }

    public function sendBulk(Request $request, NotificationGateway $gateway)
    {
        $data = $request->validate([
            'scope' => ['required', 'in:all_users,selected_families'],
            'families' => ['array'],
            'families.*' => ['integer'],
            'channels' => ['required', 'array'],
            'channels.*' => ['in:email,sms'],
            'email_subject' => ['nullable', 'string', 'max:255'],
            'email_body' => ['nullable', 'string'],
            'sms_body' => ['nullable', 'string', 'max:1000'],
        ]);

        $sendEmail = in_array('email', $data['channels'], true);
        $sendSms = in_array('sms', $data['channels'], true);

        $recipients = [];
        if ($data['scope'] === 'all_users') {
            $recipients = User::query()->with('person')->get();
        } else {
            $familyIds = collect($data['families'] ?? [])->map(fn($id) => (int)$id)->all();
            $families = Family::query()->whereIn('id', $familyIds)->get();
            $users = collect();
            foreach ($families as $family) {
                $family->loadMissing('members.user');
                foreach ($family->members as $person) {
                    if ($person->user) {
                        $users->push($person->user);
                    }
                }
            }
            $recipients = $users->unique('id')->values();
        }

        $emailSubject = (string) ($data['email_subject'] ?? '');
        $emailBody = (string) ($data['email_body'] ?? '');
        $smsBody = (string) ($data['sms_body'] ?? '');

        $sentEmail = 0;
        $sentSms = 0;
        foreach ($recipients as $user) {
            $person = $user->person;
            $familyName = $person?->families()->value('name');
            $vars = [
                '{name}' => $user->name ?? ($person?->first_name ?: 'Friend'),
                '{family_name}' => $familyName ?? 'Your Family',
                '{app_name}' => config('app.name', 'FamilyTree'),
            ];

            if ($sendEmail && $user->email && $emailSubject && $emailBody) {
                $content = strtr($emailBody, $vars);
                try {
                    $gateway->sendEmail($user->email, strtr($emailSubject, $vars), 'emails.admin_bulk_notification', [
                        'content_html' => nl2br(e($content)),
                        'appName' => config('app.name', 'FamilyTree'),
                    ]);
                    $sentEmail++;
                } catch (\Throwable $e) {
                    // swallow errors to continue sending to others
                }
            }

            if ($sendSms && $smsBody) {
                $phone = is_array($person?->meta ?? null) ? ($person->meta['phone'] ?? null) : null;
                if ($phone) {
                    try {
                        $gateway->sendSms($phone, strtr($smsBody, $vars));
                        $sentSms++;
                    } catch (\Throwable $e) {
                        // ignore sms failure per recipient
                    }
                }
            }
        }

        return back()->with('status', "Bulk notification queued. Emails: {$sentEmail}, SMS: {$sentSms}.");
    }
}
