<?php

namespace App\Http\Controllers;

use App\Models\ActivityLog;
use App\Models\Family;
use App\Models\FamilyWallPost;
use App\Models\Person;
use App\Models\Relationship;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\DB;

class AnalyticsController extends Controller
{
    public function index(Request $request)
    {
        $overview = $this->overviewMetrics();
        $dailyActivity = $this->dailyActivityTrend();
        $engagementLeaders = $this->engagementLeaders();
        $relationshipBreakdown = $this->relationshipBreakdown();
        $activityFeed = $this->activityFeed();

        return view('analytics.index', compact(
            'overview',
            'dailyActivity',
            'engagementLeaders',
            'relationshipBreakdown',
            'activityFeed'
        ));
    }

    private function overviewMetrics(): array
    {
        $now = Carbon::now();
        $monthStart = $now->copy()->subMonth();

        $familyCount = Family::count();
        $newFamilies = Family::where('created_at', '>=', $monthStart)->count();

        $memberCount = Person::count();
        $newMembers = Person::where('created_at', '>=', $monthStart)->count();

        $relationshipCount = Relationship::count();
        $newRelationships = Relationship::where('created_at', '>=', $monthStart)->count();

        $wallPostCount = FamilyWallPost::count();
        $recentPosts = FamilyWallPost::where('created_at', '>=', $monthStart)->count();

        return [
            [
                'label' => 'Families',
                'value' => number_format($familyCount),
                'delta' => '+'.number_format($newFamilies).' this month',
                'icon' => 'diversity_3',
            ],
            [
                'label' => 'Members',
                'value' => number_format($memberCount),
                'delta' => '+'.number_format($newMembers).' joined',
                'icon' => 'groups',
            ],
            [
                'label' => 'Relationships',
                'value' => number_format($relationshipCount),
                'delta' => '+'.number_format($newRelationships).' connected',
                'icon' => 'hub',
            ],
            [
                'label' => 'Wall Posts',
                'value' => number_format($wallPostCount),
                'delta' => number_format($recentPosts).' last 30 days',
                'icon' => 'dynamic_feed',
            ],
        ];
    }

    private function dailyActivityTrend(int $days = 14): Collection
    {
        $startDate = Carbon::now()->subDays($days - 1)->startOfDay();

        $activity = ActivityLog::query()
            ->selectRaw('DATE(created_at) as day, COUNT(*) as total')
            ->where('created_at', '>=', $startDate)
            ->groupBy('day')
            ->orderBy('day')
            ->get()
            ->mapWithKeys(fn ($row) => [$row->day => (int) $row->total]);

        return collect(range(0, $days - 1))->map(function (int $offset) use ($startDate, $activity) {
            $day = (clone $startDate)->addDays($offset);
            $key = $day->format('Y-m-d');

            return [
                'label' => $day->format('M j'),
                'value' => $activity[$key] ?? 0,
            ];
        });
    }

    private function engagementLeaders(int $limit = 5): Collection
    {
        $monthStart = Carbon::now()->subMonth();

        return Family::query()
            ->withCount([
                'members',
                'wallPosts',
                'wallPosts as posts_last_30_count' => fn ($query) => $query->where('created_at', '>=', $monthStart),
            ])
            ->orderByDesc('members_count')
            ->take($limit)
            ->get()
            ->map(function (Family $family) {
                return [
                    'name' => $family->name,
                    'members' => number_format($family->members_count),
                    'posts' => number_format($family->wall_posts_count),
                    'recent_posts' => number_format($family->posts_last_30_count ?? 0),
                ];
            });
    }

    private function relationshipBreakdown(): Collection
    {
        return Relationship::query()
            ->select('type', DB::raw('COUNT(*) as total'))
            ->groupBy('type')
            ->orderByDesc('total')
            ->get()
            ->map(function ($row) {
                return [
                    'type' => ucfirst($row->type),
                    'count' => (int) $row->total,
                ];
            });
    }

    private function activityFeed(int $limit = 10): Collection
    {
        return ActivityLog::query()
            ->with(['user:id,name', 'family:id,name'])
            ->latest('created_at')
            ->take($limit)
            ->get()
            ->map(function (ActivityLog $log) {
                return [
                    'event' => ucwords(str_replace(['.', '_'], [' ', ' '], $log->event)),
                    'family' => $log->family?->name,
                    'actor' => $log->user?->name ?? 'System',
                    'time' => optional($log->created_at)->diffForHumans() ?? '—',
                ];
            });
    }
}
