<?php

namespace App\Http\Controllers;

use App\Models\Family;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Validation\Rule;
use Illuminate\Support\Str;

class FamilyController extends Controller
{
    public function index(Request $request)
    {
        $families = Family::query()->with('currentSubscription.plan')->latest()->paginate(15);
        return response()->json($families);
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'slug' => ['nullable', 'string', 'max:255', 'unique:families,slug'],
            'description' => ['nullable', 'string'],
            'settings' => ['nullable', 'array'],
        ]);

        if (empty($data['slug'])) {
            $data['slug'] = Str::slug($data['name']) . '-' . Str::random(6);
        }

        $data['created_by'] = Auth::id();

        $family = Family::create($data);

        return response()->json($family, 201);
    }

    public function show(Family $family)
    {
        return response()->json($family->load('currentSubscription.plan'));
    }

    public function update(Request $request, Family $family)
    {
        $data = $request->validate([
            'name' => ['sometimes', 'string', 'max:255'],
            'slug' => ['sometimes', 'string', 'max:255', Rule::unique('families', 'slug')->ignore($family->id)],
            'description' => ['nullable', 'string'],
            'settings' => ['nullable', 'array'],
        ]);

        $family->update($data);

        return response()->json($family);
    }

    public function destroy(Family $family)
    {
        $family->delete();
        return response()->json(['message' => 'Deleted']);
    }
}
