<?php

namespace App\Notifications;

use App\Models\Family;
use App\Models\FamilyEvent;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Notifications\Messages\MailMessage;
use Illuminate\Notifications\Notification;

class FamilyEventReminderNotification extends Notification implements ShouldQueue
{
    use Queueable;

    public function __construct(public Family $family, public FamilyEvent $event)
    {
        $this->onQueue('notifications');
    }

    public function via(object $notifiable): array
    {
        $channels = ['database'];
        if (method_exists($notifiable, 'getEmailForPasswordReset') || !empty($notifiable->email)) {
            $channels[] = 'mail';
        }
        return $channels;
    }

    public function toMail(object $notifiable): MailMessage
    {
        return (new MailMessage)
            ->subject('Upcoming family event: ' . $this->event->title)
            ->greeting('Hello ' . ($notifiable->name ?? ''))
            ->line('An event is coming up in ' . $this->family->name . '.')
            ->line('Title: ' . $this->event->title)
            ->line('When: ' . optional($this->event->starts_at)->format('M d, Y H:i'))
            ->line(($this->event->location ? ('Location: ' . $this->event->location) : ''))
            ->action('View Events', route('families.events.index', $this->family->id))
            ->line('— FamilyTree');
    }

    public function toArray(object $notifiable): array
    {
        return [
            'type' => 'family_event_reminder',
            'family_id' => $this->family->id,
            'family_name' => $this->family->name,
            'event_id' => $this->event->id,
            'title' => $this->event->title,
            'starts_at' => optional($this->event->starts_at)->toIso8601String(),
        ];
    }
}
